-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 19.11.2025

MudSystemSettings = {}
MudSystemSettings.name = g_currentModName or "MudSystemSettings"
MudSystemSettings.path = g_currentModDirectory or ""

MudSystemSettings.SETTINGS = {}
MudSystemSettings.CONTROLS = {}

addModEventListener(MudSystemSettings)

MudSystemSettings._mpWantsRequest = false
MudSystemSettings._mpRequestSent  = false

-- ---------------------------------------------------------
-- CONFIG
-- ---------------------------------------------------------

MudSystemSettings.xmlKey  = "mudSystemPhysicsSettings"
MudSystemSettings.xmlFile = "modSettings/MudSystemPhysics.xml"

local function clamp(x, mn, mx)
    if x == nil then return mn end
    if x < mn then return mn end
    if x > mx then return mx end
    return x
end

local function boolTo01(b) return b and 1 or 0 end
local function int01ToBool(v) return (tonumber(v) or 0) >= 0.5 end

local function fmtFloat(v, decimals)
    decimals = decimals or 2
    return string.format("%0." .. tostring(decimals) .. "f", v)
end

function MudSystemSettings:buildRange(id, minV, maxV, step, decimals)
    local setting = self.SETTINGS[id]
    if setting ~= nil and setting.values ~= nil and setting.strings ~= nil then
        return setting.values, setting.strings
    end

    local values, strings = {}, {}
    local n = math.floor((maxV - minV) / step + 0.5)
    for i = 0, n do
        local v = minV + i * step
        v = clamp(v, minV, maxV)
        values[#values + 1] = v
        strings[#strings + 1] = fmtFloat(v, decimals)
    end

    self.SETTINGS[id] = self.SETTINGS[id] or {}
    self.SETTINGS[id].values = values
    self.SETTINGS[id].strings = strings
    return values, strings
end

function MudSystemSettings:getTextSafe(key, fallback)
    if g_i18n ~= nil and g_i18n:hasText(key) then
        return g_i18n:getText(key)
    end
    return fallback or key
end

-----------------------------------------------------------
-- ITEMS
-----------------------------------------------------------
MudSystemSettings.items = {
	-- Reset
    -- IMPORTANT: this is NOT drawn as a toggle anymore.
    -- We keep this item as a hidden placeholder (for MP snapshot index stability),
    -- while the actual reset is exposed as a bottom-bar BUTTON like RealisticWeather.
    { id="mudsys_resetAll",             section="mud", type="bool",  target="MudSystemSettings", key="resetAll" },
    -- MudPhysics
    { id="mp_enabled",                  section="mud", type="bool",  target="MudPhysics", key="enabled" },
    { id="mp_freezeMudByTempEnable",    section="mud", type="bool",  target="MudPhysics", key="freezeMudByTempEnable" },
    { id="mp_extraWheelSinkEnable",     section="mud", type="bool",  target="MudPhysics", key="extraWheelSinkEnable" },

    -- NEW: winter slip
    { id="mp_winterSlipEnable",         section="mud", type="bool",  target="MudPhysics", key="winterSlipEnable" },
    { id="mp_winterSlipTempC",          section="mud", type="float", target="MudPhysics", key="winterSlipTempC",        min=-10.0, max=0.0,  step=1.0,  decimals=0 },
    { id="mp_winterSlipMul",            section="mud", type="float", target="MudPhysics", key="winterSlipMul",          min=0.0,   max=1.0,  step=0.01, decimals=2 },

    -- NEW: normal slip
    { id="mp_normalSlipEnable",         section="mud", type="bool",  target="MudPhysics", key="normalSlipEnable" },
    { id="mp_normalSlipMul",            section="mud", type="float", target="MudPhysics", key="normalSlipMul",          min=0.0,   max=1.0,  step=0.01, decimals=2 },

    -- NEW: rain slip
    { id="mp_rainSlipEnable",           section="mud", type="bool",  target="MudPhysics", key="rainSlipEnable" },
    { id="mp_rainSlipWetnessMin",       section="mud", type="float", target="MudPhysics", key="rainSlipWetnessMin",     min=0.0,   max=1.0,  step=0.01, decimals=2 },
    { id="mp_rainSlipMul",              section="mud", type="float", target="MudPhysics", key="rainSlipMul",            min=0.0,   max=1.0,  step=0.01, decimals=2 },
    { id="mp_rainSlipMaxMul",           section="mud", type="float", target="MudPhysics", key="rainSlipMaxMul",         min=0.0,   max=1.0,  step=0.01, decimals=2 },

    -- perma stuck (старое было bool, добавляем шанс)
    { id="mp_permaStuckEnable",         section="mud", type="bool",  target="MudPhysics", key="permaStuckEnable" },
    { id="mp_permaStuckChanceOnStruggle", section="mud", type="float", target="MudPhysics", key="permaStuckChanceOnStruggle", min=0.0, max=0.2, step=0.001, decimals=2 },

    { id="mp_motorLoadEnable",          section="mud", type="bool",  target="MudPhysics", key="motorLoadEnable" },
    { id="mp_particlesEnable",          section="mud", type="bool",  target="MudPhysics", key="particlesEnable" },
    { id="mp_extraParticlesEnable",     section="mud", type="bool",  target="MudPhysics", key="extraParticlesEnable" },

    -- NEW: mud variations/bob
    { id="mp_mudVarStrength",           section="mud", type="float", target="MudPhysics", key="mudVarStrength",         min=0.0,   max=1.0,  step=0.01, decimals=2 },
    { id="mp_mudVarCell",               section="mud", type="float", target="MudPhysics", key="mudVarCell",             min=0.0,   max=10.0, step=0.01, decimals=2 },
    { id="mp_mudBobAmp",                section="mud", type="float", target="MudPhysics", key="mudBobAmp",             min=0.0,   max=1.0,  step=0.01, decimals=2 },

    -- NEW: extra particle controls
    { id="mp_extraParticleOnlyWetMud",  section="mud", type="bool",  target="MudPhysics", key="extraParticleOnlyWetMud" },
    { id="mp_extraParticleOffsetY",     section="mud", type="float", target="MudPhysics", key="extraParticleOffsetY",  min=-1.0,  max=1.0,  step=0.01, decimals=2 },

    -- dirt
    { id="mp_dirtEnable",               section="mud", type="bool",  target="MudPhysics", key="dirtEnable" },
    { id="mp_dirtMinEffMud",            section="mud", type="float", target="MudPhysics", key="dirtMinEffMud",          min=0.0,   max=1.0,  step=0.01,  decimals=2 },
    { id="mp_dirtWetnessMin",           section="mud", type="float", target="MudPhysics", key="dirtWetnessMin",         min=0.0,   max=1.0,  step=0.01,  decimals=2 },
    { id="mp_dirtBodyPerSec",           section="mud", type="float", target="MudPhysics", key="dirtBodyPerSec",         min=0.0,   max=1.0,  step=0.001, decimals=3 },
    { id="mp_dirtWheelPerSec",          section="mud", type="float", target="MudPhysics", key="dirtWheelPerSec",        min=0.0,   max=1.0,  step=0.001, decimals=3 },

    -- остальное старое
    { id="mp_wheelBrakeEnable",         section="mud", type="bool",  target="MudPhysics", key="wheelBrakeEnable" },

    { id="mp_sinkInSpeed",              section="mud", type="float", target="MudPhysics", key="sinkInSpeed",            min=0.0, max=10.0, step=0.01, decimals=2 },
    { id="mp_sinkOutSpeed",             section="mud", type="float", target="MudPhysics", key="sinkOutSpeed",           min=0.0, max=10.0, step=0.01, decimals=2 },
    { id="mp_radiusMinFactor",          section="mud", type="float", target="MudPhysics", key="radiusMinFactor",        min=0.0, max=1.0,  step=0.01, decimals=2 },

    { id="mp_emitMultWetMud",           section="mud", type="float", target="MudPhysics", key="emitMultWetMud",         min=0.0, max=50.0, step=0.01, decimals=2 },
    { id="mp_sizeMultWetMud",           section="mud", type="float", target="MudPhysics", key="sizeMultWetMud",         min=0.0, max=50.0, step=0.01, decimals=2 },
    { id="mp_speedMultWetMud",          section="mud", type="float", target="MudPhysics", key="speedMultWetMud",        min=0.0, max=50.0, step=0.01, decimals=2 },

    { id="mp_wheelBrakeBase",           section="mud", type="float", target="MudPhysics", key="wheelBrakeBase",         min=0.0, max=15.0, step=0.01, decimals=2 },
    { id="mp_wheelBrakeFromSink",       section="mud", type="float", target="MudPhysics", key="wheelBrakeFromSink",     min=0.0, max=15.0, step=0.01, decimals=2 },
    { id="mp_wheelBrakeFromSlip",       section="mud", type="float", target="MudPhysics", key="wheelBrakeFromSlip",     min=0.0, max=15.0, step=0.01, decimals=2 },

    

    -- FieldGroundMudPhysics
    { id="fg_enabled",                  section="field", type="bool",  target="FieldGroundMudPhysics", key="enabled" },
    { id="fg_motorLoadEnable",          section="field", type="bool",  target="FieldGroundMudPhysics", key="motorLoadEnable" },
    { id="fg_wheelBrakeEnable",         section="field", type="bool",  target="FieldGroundMudPhysics", key="wheelBrakeEnable" },

    -- NEW: radius sink block
    { id="fg_radiusSinkEnable",         section="field", type="bool",  target="FieldGroundMudPhysics", key="radiusSinkEnable" },
    { id="fg_radiusMinFactor",          section="field", type="float", target="FieldGroundMudPhysics", key="radiusMinFactor",     min=0.0, max=1.0,  step=0.01, decimals=2 },
    { id="fg_radiusSinkInSpeed",        section="field", type="float", target="FieldGroundMudPhysics", key="radiusSinkInSpeed",   min=0.0, max=2.0, step=0.001, decimals=3 },
    { id="fg_radiusSinkOutSpeed",       section="field", type="float", target="FieldGroundMudPhysics", key="radiusSinkOutSpeed",  min=0.0, max=2.0, step=0.001, decimals=3 },

    -- NEW: slip min/max
    { id="fg_slipMinMul",               section="field", type="float", target="FieldGroundMudPhysics", key="slipMinMul",          min=0.0, max=1.0, step=0.01, decimals=2 },
    { id="fg_slipMaxMul",               section="field", type="float", target="FieldGroundMudPhysics", key="slipMaxMul",          min=0.0, max=1.0, step=0.01, decimals=2 },

    { id="fg_extraParticlesEnable",     section="field", type="bool",  target="FieldGroundMudPhysics", key="extraParticlesEnable" },
    -- NEW: erase fruit to zero under wheels (server)
    { id="fg_eraseFruitEnable",         section="field", type="bool",  target="FieldGroundMudPhysics", key="eraseFruitEnable" },

	{ id="fg_eraseFoliageOnlyFields",   section="field", type="bool",  target="FieldGroundMudPhysics", key="eraseFoliageOnlyFields" },

    -- NEW: dirt tuning for field module
    { id="fg_dirtEnable",               section="field", type="bool",  target="FieldGroundMudPhysics", key="dirtEnable" },
    { id="fg_dirtMinEffMud",            section="field", type="float", target="FieldGroundMudPhysics", key="dirtMinEffMud",        min=0.0, max=1.0, step=0.01,  decimals=2 },
    { id="fg_dirtWetnessMin",           section="field", type="float", target="FieldGroundMudPhysics", key="dirtWetnessMin",       min=0.0, max=1.0, step=0.01,  decimals=2 },
    { id="fg_dirtBodyPerSec",           section="field", type="float", target="FieldGroundMudPhysics", key="dirtBodyPerSec",       min=0.0, max=1.0, step=0.001, decimals=3 },
    { id="fg_dirtWheelPerSec",          section="field", type="float", target="FieldGroundMudPhysics", key="dirtWheelPerSec",      min=0.0, max=1.0, step=0.001, decimals=3 },

    { id="fg_wheelBrakeBase",           section="field", type="float", target="FieldGroundMudPhysics", key="wheelBrakeBase",       min=0.0, max=15.0, step=0.01, decimals=2 },
    { id="fg_wheelBrakeFromSink",       section="field", type="float", target="FieldGroundMudPhysics", key="wheelBrakeFromSink",   min=0.0, max=15.0, step=0.01, decimals=2 },
    { id="fg_wheelBrakeFromSlip",       section="field", type="float", target="FieldGroundMudPhysics", key="wheelBrakeFromSlip",   min=0.0, max=15.0, step=0.01, decimals=2 },
}

function MudSystemSettings:getTargetTable(targetName)
    if targetName == "MudPhysics" then
        return rawget(_G, "MudPhysics")
    elseif targetName == "FieldGroundMudPhysics" then
        return rawget(_G, "FieldGroundMudPhysics")
    elseif targetName == "MudSystemSettings" then
        return self
    end
    return nil
end

-- ---------------------------------------------------------
-- XML READ/WRITE
-- ---------------------------------------------------------

function MudSystemSettings:getUserSettingsPath()
    return Utils.getFilename(self.xmlFile, getUserProfileAppPath())
end

function MudSystemSettings:writeSettings()
    local userSettingsFile = self:getUserSettingsPath()
    local xmlFile = createXMLFile("settings", userSettingsFile, self.xmlKey)
    if xmlFile == 0 then
        print("[MudSystemSettings] Failed to create XML for saving settings")
        return
    end

    for _, item in ipairs(self.items) do
        if item.id ~= "mudsys_resetAll" then
            local tgt = self:getTargetTable(item.target)
            if tgt ~= nil then
                local value = tgt[item.key]
                local k = string.format("%s.%s#value", self.xmlKey, item.id)
                if item.type == "bool" then
                    setXMLInt(xmlFile, k, boolTo01(value == true))
                else
                    setXMLFloat(xmlFile, k, tonumber(value) or 0)
                end
            end
        end
    end

    saveXMLFile(xmlFile)
    delete(xmlFile)
end

function MudSystemSettings:readSettings()
    local userSettingsFile = self:getUserSettingsPath()
    if not fileExists(userSettingsFile) then
        self:writeSettings()
        return
    end

    local xmlFile = loadXMLFile(self.xmlKey, userSettingsFile)
    if xmlFile == 0 then
        print("[MudSystemSettings] Failed to load XML, keeping defaults")
        return
    end

    for _, item in ipairs(self.items) do
        if item.id ~= "mudsys_resetAll" then
            local tgt = self:getTargetTable(item.target)
            if tgt ~= nil then
                local k = string.format("%s.%s#value", self.xmlKey, item.id)
                if hasXMLProperty(xmlFile, k) then
                    if item.type == "bool" then
                        tgt[item.key] = int01ToBool(getXMLInt(xmlFile, k))
                    else
                        local v = getXMLFloat(xmlFile, k)
                        if v ~= nil then
                            tgt[item.key] = clamp(v, item.min or -math.huge, item.max or math.huge)
                        end
                    end
                end
            end
        end
    end

    delete(xmlFile)
end

-- ---------------------------------------------------------
-- MENU INTEGRATION
-- ---------------------------------------------------------

function MudSystemSettings:getStateIndexFromValue(id, value)
    local setting = self.SETTINGS[id]
    if setting == nil or setting.values == nil then
        return 1
    end
    local values = setting.values

    if type(value) == "number" then
        local bestIdx, bestDiff = 1, math.huge
        for i, v in ipairs(values) do
            local d = math.abs(v - value)
            if d < bestDiff then
                bestDiff = d
                bestIdx = i
            end
        end
        return bestIdx
    else
        for i, v in ipairs(values) do
            if v == value then
                return i
            end
        end
        return 1
    end
end

function MudSystemSettings:updateFocusIds(element)
    if not element then return end
    element.focusId = FocusManager:serveAutoFocusId()
    for _, child in pairs(element.elements) do
        self:updateFocusIds(child)
    end
end

function MudSystemSettings:ensureSettingDefinition(item)
    if item.type == "bool" then
        self.SETTINGS[item.id] = {
            values  = { false, true },
            strings = { self:getTextSafe("mudsys_option_off", "OFF"), self:getTextSafe("mudsys_option_on", "ON") }
        }
        return
    end

    self:buildRange(item.id, item.min, item.max, item.step, item.decimals)
end

function MudSystemSettings:addMenuSection(settingsLayout, settingsPage, titleKey, titleFallback)
    local header
    for _, elem in ipairs(settingsLayout.elements) do
        if elem.name == "sectionHeader" then
            header = elem:clone(settingsLayout)
            break
        end
    end

    if header == nil then
        header = TextElement.new()
        header:applyProfile("fs25_settingsSectionHeader", true)
        header.name = "sectionHeader"
        settingsLayout:addElement(header)
    end

    header:setText(self:getTextSafe(titleKey, titleFallback))
    header.focusId = FocusManager:serveAutoFocusId()
    table.insert(settingsPage.controlsList, header)
    self.CONTROLS[titleKey] = header
end

function MudSystemSettings:addMenuOption(item)
    local inGameMenu = g_gui.screenControllers[InGameMenu]
    local settingsPage = inGameMenu.pageSettings
    local settingsLayout = settingsPage.generalSettingsLayout

    self:ensureSettingDefinition(item)
    local def = self.SETTINGS[item.id]

    local template = (#def.values == 2) and settingsPage.checkWoodHarvesterAutoCutBox or settingsPage.multiVolumeVoiceBox
    local menuOptionBox = template:clone(settingsLayout)
    menuOptionBox.id = item.id .. "Box"

    local menuOption = menuOptionBox.elements[1]
    menuOption.id = item.id
    menuOption.target = self
    menuOption:setCallback("onClickCallback", "onMenuOptionChanged")
    menuOption:setDisabled(false)

    local toolTip = menuOption.elements[1]
    toolTip:setText(self:getTextSafe("tooltip_" .. item.id, ""))

    local label = menuOptionBox.elements[2]
    label:setText(self:getTextSafe("setting_" .. item.id, item.id))

    menuOption:setTexts({ unpack(def.strings) })

    if item.id == "mudsys_resetAll" then
        menuOption:setState(1)
    else
        local tgt = self:getTargetTable(item.target)
        local currentValue = tgt and tgt[item.key]
        if item.type == "bool" then
            menuOption:setState((currentValue == true) and 2 or 1)
        else
            menuOption:setState(self:getStateIndexFromValue(item.id, tonumber(currentValue) or 0))
        end
    end

    self.CONTROLS[item.id] = menuOption
    self:updateFocusIds(menuOptionBox)
    table.insert(settingsPage.controlsList, menuOptionBox)
end

function MudSystemSettings:captureDefaults()
    self.DEFAULTS = self.DEFAULTS or {}
    for _, item in ipairs(self.items or {}) do
        if item.id ~= "mudsys_resetAll" then
            local tgt = self:getTargetTable(item.target)
            if tgt ~= nil then
                local v = tgt[item.key]
                if item.type == "bool" then
                    self.DEFAULTS[item.id] = (v == true)
                else
                    self.DEFAULTS[item.id] = tonumber(v) or 0
                end
            end
        end
    end
end

function MudSystemSettings:resetToDefaults()
    if self.DEFAULTS == nil then
        self:captureDefaults()
    end

    for _, item in ipairs(self.items or {}) do
        if item.id ~= "mudsys_resetAll" then
            local defV = self.DEFAULTS[item.id]
            if defV ~= nil then
                self:applyOneItemValue(item, defV, false)
            end
        end
    end

    self:writeSettings()
end

function MudSystemSettings:refreshMenuStates()
    local isAdmin = (g_currentMission ~= nil) and (g_currentMission:getIsServer() or g_currentMission.isMasterUser)

    for _, item in ipairs(self.items or {}) do
        local menuOption = self.CONTROLS[item.id]
        if menuOption ~= nil then
            if item.id == "mudsys_resetAll" then
                menuOption:setState(1)
            else
                local tgt = self:getTargetTable(item.target)
                local currentValue = tgt and tgt[item.key]

                if item.type == "bool" then
                    menuOption:setState((currentValue == true) and 2 or 1)
                else
                    menuOption:setState(self:getStateIndexFromValue(item.id, tonumber(currentValue) or 0))
                end
            end

            menuOption:setDisabled(not isAdmin)
        end
    end
end

function MudSystemSettings:onMenuOptionChanged(state, menuOption)
    local id = menuOption.id

    if id == "mudsys_resetAll" then
        local isAdmin = (g_currentMission ~= nil) and (g_currentMission:getIsServer() or g_currentMission.isMasterUser == true)
        if isAdmin then
            self:resetToDefaults()
            self:refreshMenuStates()
            if MudSystemSettingsSyncEvent ~= nil and MudSystemSettingsSyncEvent.sendSnapshot ~= nil then
                MudSystemSettingsSyncEvent.sendSnapshot(self:collectSnapshot())
            end
        end

        if menuOption ~= nil then
            menuOption:setState(1)
        end
        return
    end

    local item
    for _, it in ipairs(self.items) do
        if it.id == id then item = it break end
    end
    if item == nil then return end

    local def = self.SETTINGS[id]
    local value = def.values[state]

    self:applyOneItemValue(item, value, true)

    if g_currentMission ~= nil and g_currentMission.missionDynamicInfo ~= nil then
        local isAdmin = g_currentMission:getIsServer() or (g_currentMission.isMasterUser == true)
        if isAdmin and MudSystemSettingsSyncEvent ~= nil and MudSystemSettingsSyncEvent.sendSnapshot ~= nil then
            MudSystemSettingsSyncEvent.sendSnapshot(self:collectSnapshot())
        end
    end
end

function MudSystemSettings:applyOneItemValue(item, value, doWrite)
    local tgt = self:getTargetTable(item.target)
    if tgt ~= nil then
        if item.type == "bool" then
            tgt[item.key] = (value == true)
        else
            tgt[item.key] = clamp(tonumber(value) or 0, item.min or -math.huge, item.max or math.huge)
        end
    end

    if doWrite == true then
        self:writeSettings()
    end

    if tgt ~= nil and type(tgt.onSettingsChanged) == "function" then
        tgt:onSettingsChanged()
    end
end

function MudSystemSettings:collectSnapshot()
    local snap = {}
    for i, item in ipairs(self.items) do
        if item.id == "mudsys_resetAll" then
            snap[i] = false
        else
            local tgt = self:getTargetTable(item.target)
            local v = tgt and tgt[item.key]
            if item.type == "bool" then
                snap[i] = (v == true)
            else
                snap[i] = tonumber(v) or 0
            end
        end
    end
    return snap
end

function MudSystemSettings:applySnapshot(snapshot, doWrite)
    if snapshot == nil then return end

    for i, item in ipairs(self.items) do
        if item.id ~= "mudsys_resetAll" then
            local v = snapshot[i]
            if item.type == "bool" then
                self:applyOneItemValue(item, v == true, false)
            else
                self:applyOneItemValue(item, tonumber(v) or 0, false)
            end
        end
    end

    if doWrite == true then
        self:writeSettings()
    end
end

function MudSystemSettings:buildMenu()
    if g_gui == nil or g_gui.screenControllers == nil then
        return false
    end
    local inGameMenu = g_gui.screenControllers[InGameMenu]
    if inGameMenu == nil or inGameMenu.pageSettings == nil then
        return false
    end

    local settingsPage = inGameMenu.pageSettings
    local settingsLayout = settingsPage.generalSettingsLayout
    if settingsLayout == nil then
        return false
    end

    self:addMenuSection(settingsLayout, settingsPage, "menu_MudSystemPhysics_TITLE", "Mud System Physics")
    for _, item in ipairs(self.items) do
        if item.section == "mud" then
            self:addMenuOption(item)
        end
    end

    self:addMenuSection(settingsLayout, settingsPage, "menu_FieldGroundMudPhysics_TITLE", "Field Ground Mud Physics")
    for _, item in ipairs(self.items) do
        if item.section == "field" then
            self:addMenuOption(item)
        end
    end

    settingsLayout:invalidateLayout()
    return true
end

-- ---------------------------------------------------------
-- LIFECYCLE
-- ---------------------------------------------------------

function MudSystemSettings:loadMap(name)
    self:captureDefaults()
    self:readSettings()

    if g_currentMission ~= nil then
        FSBaseMission.registerToLoadOnMapFinished(g_currentMission, self)
    end
end

function MudSystemSettings:onLoadMapFinished()
    local isServer = (g_currentMission ~= nil) and g_currentMission:getIsServer()

    if isServer then
        self:readSettings()
    end

    self:buildMenu()

    if not isServer then
        self._mpWantsRequest = true
        self._mpRequestSent  = false
    end
end

function MudSystemSettings:onUpdate(dt)
    if self._mpWantsRequest and not self._mpRequestSent then
        if g_client ~= nil and g_client.getServerConnection ~= nil and g_client:getServerConnection() ~= nil then
            if MudSystemSettingsSyncEvent ~= nil and MudSystemSettingsSyncEvent.sendRequest ~= nil then
                MudSystemSettingsSyncEvent.sendRequest()
            end
            self._mpRequestSent  = true
            self._mpWantsRequest = false
        end
    end
end

-- ---------------------------------------------------------
-- RESET BUTTON (bottom bar) like FS25_RealisticWeather
-- ---------------------------------------------------------

function MudSystemSettings:doResetFromButton()
    local isAdmin = (g_currentMission ~= nil) and (g_currentMission:getIsServer() or g_currentMission.isMasterUser == true)
    if not isAdmin then
        return
    end

    local function doReset()
        self:resetToDefaults()
        self:refreshMenuStates()

        if MudSystemSettingsSyncEvent ~= nil and MudSystemSettingsSyncEvent.sendSnapshot ~= nil then
            MudSystemSettingsSyncEvent.sendSnapshot(self:collectSnapshot())
        end
    end

    -- Confirmation dialog (like RealisticWeather rebuild button flow)
    if g_gui ~= nil and g_gui.showYesNoDialog ~= nil then
        local title = self:getTextSafe("mudsys_ui_resetTitle", self:getTextSafe("ui_warning", "Warning"))
        local text  = self:getTextSafe("mudsys_ui_resetConfirm", "Reset MudSystem settings to defaults?")

        g_gui:showYesNoDialog({
            title = title,
            text = text,
            callback = function(_, yes)
                if yes then
                    doReset()
                end
            end
        })
    else
        -- Fallback (shouldn't happen): reset immediately
        doReset()
    end
end

function MudSystemSettings:_pickFreeSettingsExtraAction(settingsFrame)
    local used = {}

    if settingsFrame ~= nil and settingsFrame.menuButtonInfo ~= nil then
        for _, b in ipairs(settingsFrame.menuButtonInfo) do
            if b ~= nil and b.inputAction ~= nil then
                used[b.inputAction] = true
            end
        end
    end

    local candidates = {
        InputAction.MENU_EXTRA_1,
        InputAction.MENU_EXTRA_2,
        InputAction.MENU_EXTRA_3,
        InputAction.MENU_EXTRA_4
    }

    for _, a in ipairs(candidates) do
        if a ~= nil and used[a] ~= true then
            return a
        end
    end

    -- If all extras are taken by other mods, don't steal their hotkey.
    -- We still show the button (mouse clickable) without an inputAction.
    return nil
end

function MudSystemSettings:ensureResetButton(settingsFrame)
    if settingsFrame == nil or settingsFrame.menuButtonInfo == nil then
        return
    end

    self._resetButton = self._resetButton or {
        inputAction = self:_pickFreeSettingsExtraAction(settingsFrame),
        text = self:getTextSafe("mudsys_ui_resetAll", "Reset MudSystem settings"),
        callback = function()
            if MudSystemSettings ~= nil then
                MudSystemSettings:doResetFromButton()
            end
        end,
        showWhenPaused = true
    }

    -- Make sure we don't conflict with other mods:
    -- pick a free MENU_EXTRA_* action based on current menuButtonInfo.
    local desiredAction = self:_pickFreeSettingsExtraAction(settingsFrame)
    if self._resetButton.inputAction ~= desiredAction then
        self._resetButton.inputAction = desiredAction
    end


    -- avoid duplicates if updateButtons is called multiple times
    for _, b in ipairs(settingsFrame.menuButtonInfo) do
        if b == self._resetButton then
            return
        end
    end

    table.insert(settingsFrame.menuButtonInfo, self._resetButton)
    settingsFrame:setMenuButtonInfoDirty()
end

if InGameMenuSettingsFrame ~= nil and Utils ~= nil and Utils.appendedFunction ~= nil then
    InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(InGameMenuSettingsFrame.onFrameOpen, function()
        if MudSystemSettings == nil then return end

        MudSystemSettings:refreshMenuStates()
    end)

    -- add bottom-bar button (not a toggle in the list)
    InGameMenuSettingsFrame.updateButtons = Utils.appendedFunction(InGameMenuSettingsFrame.updateButtons, function(self)
        if MudSystemSettings == nil then return end
        MudSystemSettings:ensureResetButton(self)
    end)
end

if FocusManager ~= nil and Utils ~= nil and Utils.appendedFunction ~= nil then
    FocusManager.setGui = Utils.appendedFunction(FocusManager.setGui, function(_, gui)
        if gui == "ingameMenuSettings" and MudSystemSettings ~= nil then
            for _, control in pairs(MudSystemSettings.CONTROLS or {}) do
                if control ~= nil and (not control.focusId or not FocusManager.currentFocusData.idToElementMapping[control.focusId]) then
                    if not FocusManager:loadElementFromCustomValues(control, nil, nil, false, false) then
                        Logging.warning("[MudSystemSettings] Could not register control %s with the focus manager", control.id or control.name or tostring(control.focusId))
                    end
                end
            end
            local settingsPage = g_gui.screenControllers[InGameMenu].pageSettings
            settingsPage.generalSettingsLayout:invalidateLayout()
        end
    end)
end
