-- RMC_BeltAnchor.lua (v1.1.0.0)
-- Adds mass scaling while strapped to reduce suspension squat. Restores on unstrap.
RMCBeltAnchor = {}
RMCBeltAnchor.OBJ_DECK_CLEARANCE = 0.04 -- meters above deck/jointNode for object anchors
RMCBeltAnchor.OBJ_HEIGHT_BIAS    = 0.00 -- additional global bias for objects (keep 0.00 unless tuning)
RMCBeltAnchor.MIN_MASS_FOR_TRI   = 250  -- kg: below this, use <=2 joints when MAX=3
RMCBeltAnchor.MAX_JOINT_COUNT    = 3    -- 1..3 joints for objects
RMCBeltAnchor.OBJ_INERTIA_SZ     = 1.00 -- inertia scale Z while strapped (NON-split)
RMCBeltAnchor.OBJ_INERTIA_SY     = 1.00 -- inertia scale Y while strapped (NON-split)
RMCBeltAnchor.OBJ_INERTIA_SX     = 1.00 -- inertia scale X while strapped (NON-split)
RMCBeltAnchor.OBJ_MASS_SCALE     = 1.00 -- mass scale while strapped (NON-split)
RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY= 1    -- 1=only lower anchors; 0=force to targetY
RMCBeltAnchor.OBJ_ANCHOR_SPREAD  = 0.22 -- scales X/Z of triangle offsets for objects

RMCBeltAnchor.DEBUG = false
RMCBeltAnchor.MODE = "MULTI_JOINT" -- or "KINEMATIC_LOCK"
RMCBeltAnchor.MASS_SCALE = 0.85

RMCBeltAnchor.OFFSETS = { {0,0,0}, {0.10,0,0}, {0,0,0.10} }
RMCBeltAnchor.SPRING_K = 100000
RMCBeltAnchor.SPRING_D = 150

local function _consoleSetMode(self, mode)
    if not mode or mode == "" then
if RMCBeltAnchor.DEBUG then print(("[RMC_BeltAnchor] Mode = %s"):format(RMCBeltAnchor.MODE)) end
        return
    end
    local up = tostring(mode):upper()
    if up == "MULTI_JOINT" or up == "KINEMATIC_LOCK" then
        RMCBeltAnchor.MODE = up
if RMCBeltAnchor.DEBUG then print(("[RMC_BeltAnchor] Set mode = %s"):format(RMCBeltAnchor.MODE)) end
    else
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Invalid mode. Use MULTI_JOINT or KINEMATIC_LOCK") end
    end
end
addConsoleCommand("gsRMCBeltAnchorMode", "Get/Set RMC Belt Anchor mode (MULTI_JOINT|KINEMATIC_LOCK)", "_consoleSetMode", RMCBeltAnchor)

local function _consoleSetMassScale(self, scale)
    if scale == nil or scale == "" then
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] MASS_SCALE = %.3f", RMCBeltAnchor.MASS_SCALE)) end
        return
    end
    local v = tonumber(scale)
    if v and v > 0.1 and v <= 1.5 then
        RMCBeltAnchor.MASS_SCALE = v
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Set MASS_SCALE = %.3f", RMCBeltAnchor.MASS_SCALE)) end
    else
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Invalid MASS_SCALE. Range 0.1 .. 1.5") end
    end
end
addConsoleCommand("gsRMCBeltAnchorMassScale", "Get/Set mass scale while strapped (e.g. 0.85).", "_consoleSetMassScale", RMCBeltAnchor)

local function _hardReset()
    local veh = g_currentMission and g_currentMission.controlledVehicle
    if veh and veh.spec_tensionBelts then
        local spec = veh.spec_tensionBelts
        if veh.freeTensionBeltObject then
            for objectId, entry in pairs(spec.objectsToJoint or {}) do
                pcall(veh.freeTensionBeltObject, veh, objectId, spec.objectsToJoint, spec.isDynamic, entry and entry.object or nil)
            end
        end
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Hard reset done") end
    else
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] No controlled vehicle with tensionBelts") end
    end
end
addConsoleCommand("gsRMCBeltAnchorHardReset", "Force-free all RMC belt anchors on the controlled vehicle", "_hardReset", RMCBeltAnchor)

function RMCBeltAnchor.prerequisitesPresent(specs)
    return SpecializationUtil.hasSpecialization(TensionBelts, specs)
end

function RMCBeltAnchor.registerEventListeners(vType)
    SpecializationUtil.registerOverwrittenFunction(vType, "lockTensionBeltObject", RMCBeltAnchor.lockTensionBeltObject_OW)
    SpecializationUtil.registerOverwrittenFunction(vType, "freeTensionBeltObject", RMCBeltAnchor.freeTensionBeltObject_OW)
    SpecializationUtil.registerOverwrittenFunction(vType, "getAllowDynamicMountObjects", RMCBeltAnchor.getAllowDynamicMountObjects_OW)
    SpecializationUtil.registerEventListener(vType, "onLoad", RMCBeltAnchor)
    SpecializationUtil.registerEventListener(vType, "onDelete", RMCBeltAnchor)
    SpecializationUtil.registerEventListener(vType, "onUpdate", RMCBeltAnchor)
end

function RMCBeltAnchor:onLoad()
    -- RMC: register console command once
    if RMCBeltAnchor.CMD_INIT ~= true then
        addConsoleCommand("gsRMCObjDeckClearance", "Get/Set RMC object deck clearance (meters). Usage: gsRMCObjDeckClearance [value]", "consoleSetObjDeckClearance", RMCBeltAnchor)
        addConsoleCommand("gsRMCObjHeightBias", "Get/Set RMC object height bias (m). Usage: gsRMCObjHeightBias [value]", "consoleSetObjHeightBias", RMCBeltAnchor)
        addConsoleCommand("gsRMCObjAnchorSpread", "Get/Set RMC object anchor spread. Usage: gsRMCObjAnchorSpread [value]", "consoleSetObjAnchorSpread", RMCBeltAnchor)
        addConsoleCommand("gsRMCObjClampDownOnly", "Get/Set RMC clamp-down-only (0/1). Usage: gsRMCObjClampDownOnly [0|1]", "consoleSetObjClampDownOnly", RMCBeltAnchor)
        addConsoleCommand("gsRMCObjMassScale", "Get/Set RMC object mass scale (NON-split). Usage: gsRMCObjMassScale [value]", "consoleSetObjMassScale", RMCBeltAnchor)
        addConsoleCommand("gsRMCObjInertiaScale", "Get/Set RMC object inertia scale sx sy sz (NON-split). Usage: gsRMCObjInertiaScale sx sy sz", "consoleSetObjInertiaScale", RMCBeltAnchor)
        addConsoleCommand("gsRMCMaxJointCount", "Get/Set RMC max joint count for objects (1..3). Usage: gsRMCMaxJointCount [1..3]", "consoleSetMaxJointCount", RMCBeltAnchor)
        addConsoleCommand("gsRMCMinMassForTri", "Get/Set RMC min mass for 3-joint triangle (kg). Usage: gsRMCMinMassForTri [kg]", "consoleSetMinMassForTri", RMCBeltAnchor)
        addConsoleCommand("gsRMCPreset", "Apply RMC preset: TightLock|SmoothHaul|LightCargo. Usage: gsRMCPreset <name>", "consoleApplyPreset", RMCBeltAnchor)
        RMCBeltAnchor.CMD_INIT = true
    end
    self.spec_rmcBeltAnchor = { entries = {} }
end

function RMCBeltAnchor:onDelete()
    local s = self.spec_rmcBeltAnchor
    if not s or not s.entries then return end
    for objectId, e in pairs(s.entries) do
        if e.rmcMultiJoint and e.rmcMultiJoint.joints then
            for _,j in ipairs(e.rmcMultiJoint.joints) do pcall(removeJoint, j) end
        end
        if e.rmcMultiJoint and e.rmcMultiJoint.anchors then
            for _,a in ipairs(e.rmcMultiJoint.anchors) do if a and entityExists(a) then pcall(delete,a) end end
        end
        if e.rmcKinematic and entityExists(objectId) then
            link(getRootNode(), objectId)
            setRigidBodyType(objectId, RigidBodyType.DYNAMIC)
        end
        if e.rmcMassOrig and entityExists(objectId) and self.isServer then
            pcall(setMass, objectId, e.rmcMassOrig)
        end
    end
    self.spec_rmcBeltAnchor.entries = {}
end

-- Console: gsRMCObjDeckClearance [meters]
function RMCBeltAnchor:consoleSetObjDeckClearance(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then
            v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then
            v = tonumber(arg[1])
        end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] OBJ_DECK_CLEARANCE = %.3f m", RMCBeltAnchor.OBJ_DECK_CLEARANCE or 0))
        print("Usage: gsRMCObjDeckClearance <meters>")
        return
    end
    v = math.max(0, v)
    RMCBeltAnchor.OBJ_DECK_CLEARANCE = v
    print(string.format("[RMC_BeltAnchor] OBJ_DECK_CLEARANCE set to %.3f m", v))
end


function RMCBeltAnchor:onUpdate(dt)
    -- Mirror base TensionBelts replay queue: track if beltsToLoad is active
    local tb = self.spec_tensionBelts
    local rmc = self.spec_rmcBeltAnchor
    if tb ~= nil then
        local active = (tb.beltsToLoad ~= nil and #tb.beltsToLoad > 0) or false
        if rmc == nil then
            self.spec_rmcBeltAnchor = { entries = {}, loadQueueActive = active }
        else
            rmc.loadQueueActive = active
        end
    end
end


function RMCBeltAnchor.getAllowDynamicMountObjects_OW(self, superFunc)
    local rmc = self.spec_rmcBeltAnchor
    if rmc and rmc.blockDM then
        return false
    end
    if superFunc ~= nil then
        return superFunc(self)
    end
    return true
end


local function _newRigidJoint(jointNode, objectId, offX, offY, offZ, k, d)
    local jc = JointConstructor.new()
    jc:setActors(jointNode, objectId)

    local anchor = createTransformGroup("rmcBeltAnchor")
    link(jointNode, anchor)
    local wx, wy, wz = localToWorld(objectId, offX, offY, offZ)
    -- RMC__clamp_policy
    if getSplitType(objectId) == 0 then
        local jx, jy, jz = getWorldTranslation(jointNode)
        local clearance  = RMCBeltAnchor.OBJ_DECK_CLEARANCE or 0.04
        local targetY    = jy - clearance + (RMCBeltAnchor.OBJ_HEIGHT_BIAS or 0)
        if (RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY or 1) ~= 0 then
            if wy > targetY then wy = targetY end
        else
            wy = targetY
        end
    end


    -- RMC: deck-clamp anchors for NON-SPLIT (vehicles/objects) to reduce roll lever
    if getSplitType(objectId) == 0 then
        local jx, jy, jz = getWorldTranslation(jointNode)
        local clearance  = RMCBeltAnchor.OBJ_DECK_CLEARANCE or 0.04
        local targetY    = jy - clearance + (RMCBeltAnchor.OBJ_HEIGHT_BIAS or 0)
        if wy > targetY then wy = targetY end
    end

    setWorldTranslation(anchor, wx, wy, wz)

    jc:setJointTransforms(anchor, anchor)
    jc:setEnableCollision(true)

    jc:setRotationLimit(0, 0, 0)
    jc:setRotationLimit(1, 0, 0)
    jc:setRotationLimit(2, 0, 0)
    jc:setRotationLimitSpring(RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                              RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                              RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D)
    jc:setTranslationLimitSpring(RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                                 RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                                 RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D)

    local j = jc:finalize()
    return j, anchor
end

function RMCBeltAnchor.lockTensionBeltObject_OW(self, superFunc, objectId, objectsToJointTable, isDynamic, jointNode, object)
    -- SUPPRESS base lock entirely. One object => one joint set.
    local rmc = self.spec_rmcBeltAnchor or {}
    rmc.entries = rmc.entries or {}
    self.spec_rmcBeltAnchor = rmc

    if objectsToJointTable[objectId] ~= nil then
        return
    end
    -- RMC__preloop_setup: determine desired joint count and apply mass/inertia for NON-split
    local desiredCount = math.max(1, math.min(3, RMCBeltAnchor.MAX_JOINT_COUNT or 3))
    local okM, mass = pcall(getMass, objectId)
    if desiredCount == 3 and okM and mass and (RMCBeltAnchor.MIN_MASS_FOR_TRI or 250) > mass then
        desiredCount = 2
    end
    -- mass & inertia scaling (server-side, NON-split)
    if getSplitType(objectId) == 0 then
        local rmc = self.spec_rmcBeltAnchor or {}
        self.spec_rmcBeltAnchor = rmc
        rmc.entries = rmc.entries or {}
        local entry = rmc.entries[objectId] or {}
        rmc.entries[objectId] = entry
        if self.isServer and entityExists(objectId) then
            if entry.rmcMassOrig == nil then
                local ok, m0 = pcall(getMass, objectId)
                if ok and m0 and m0 > 0 then entry.rmcMassOrig = m0 end
            end
            local ms = RMCBeltAnchor.OBJ_MASS_SCALE or 1.0
            if entry.rmcMassOrig ~= nil and ms ~= 1.0 then
                pcall(setMass, objectId, entry.rmcMassOrig * ms)
            end
            local sx = RMCBeltAnchor.OBJ_INERTIA_SX or 1.0
            local sy = RMCBeltAnchor.OBJ_INERTIA_SY or 1.0
            local sz = RMCBeltAnchor.OBJ_INERTIA_SZ or 1.0
            if sx ~= 1.0 or sy ~= 1.0 or sz ~= 1.0 then
                pcall(setInertiaScale, objectId, sx, sy, sz)
                entry.rmcInertiaApplied = true
            end
        end
    end
    local spread = RMCBeltAnchor.OBJ_ANCHOR_SPREAD or 0.22
    local clampDownOnly = (RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY or 1) ~= 0


    -- RMC: detect if this strap call occurs during base replay queue
    local tb = self.spec_tensionBelts
    local inQueue = (tb ~= nil and tb.beltsToLoad ~= nil and #tb.beltsToLoad > 0) or false
    local rmc = self.spec_rmcBeltAnchor or {}
    self.spec_rmcBeltAnchor = rmc
    rmc.inQueue = inQueue


    -- RMC: split objects ignore our script; delegate to base strap logic
    if getSplitType(objectId) ~= 0 then
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Lock: %s  (SPLIT) -> base", tostring(self.configFileName or self.typeName))) end
        return superFunc(self, objectId, objectsToJointTable, isDynamic, jointNode, object)
    end

    -- Non-split path: enable our stiff multi-joint behavior
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Lock: %s  mode=%s  jointNode=%s", tostring(self.configFileName or self.typeName), tostring(RMCBeltAnchor.MODE), getName(jointNode))) end
    local rmc = self.spec_rmcBeltAnchor or {}
    self.spec_rmcBeltAnchor = rmc
    rmc.blockDM = true
    if object ~= nil and object.unmountKinematic ~= nil then
        pcall(function() object:unmountKinematic() end)
    end
    if self.isServer and entityExists(objectId) then
        link(getRootNode(), objectId)
        setRigidBodyType(objectId, RigidBodyType.DYNAMIC)
    end


    if RMCBeltAnchor.MODE == "KINEMATIC_LOCK" then
        if object and object.mountKinematic and (object.getSupportsMountKinematic == nil or object:getSupportsMountKinematic()) then
            object:mountKinematic(self, jointNode, 0,0,0, 0,0,0)
        else
            setRigidBodyType(objectId, RigidBodyType.KINEMATIC)
            local jx,jy,jz = getWorldTranslation(jointNode)
            local rx,ry,rz = getWorldRotation(jointNode)
            link(jointNode, objectId)
            setWorldTranslation(objectId, jx,jy,jz)
            setWorldRotation(objectId, rx,ry,rz)
        end
        objectsToJointTable[objectId] = { parent=getParent(objectId), object=object, rmcKinematic=true }
        rmc.entries[objectId] = objectsToJointTable[objectId]
        return
    end

    local created, anchors = {}, {}
    if self.isServer then
        for _,o in ipairs(RMCBeltAnchor.OFFSETS) do
            local jIdx, anc = _newRigidJoint(jointNode, objectId, o[1],o[2],o[3], RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D)
            table.insert(created, jIdx)
            table.insert(anchors, anc)
        end
        if getSplitType(objectId) ~= 0 then
            setInertiaScale(objectId,20,20,20)
        end

        -- Mass scale for non-split objects
        if getSplitType(objectId) == 0 and math.abs((RMCBeltAnchor.MASS_SCALE or 1.0) - 1.0) > 0.001 then
            local origMass = getMass(objectId) or 0
            if origMass > 0 then
                objectsToJointTable[objectId] = objectsToJointTable[objectId] or {}
                objectsToJointTable[objectId].rmcMassOrig = origMass
                local newMass = math.max(0.001, origMass * RMCBeltAnchor.MASS_SCALE)
                pcall(setMass, objectId, newMass)
            end
        end
    end

    objectsToJointTable[objectId] = objectsToJointTable[objectId] or {}
    objectsToJointTable[objectId].jointIndex = created[1]
    objectsToJointTable[objectId].jointTransform = anchors[1]
    objectsToJointTable[objectId].rmcMultiJoint = { joints=created, anchors=anchors }
    objectsToJointTable[objectId].object = object

    rmc.entries[objectId] = objectsToJointTable[objectId]
end

function RMCBeltAnchor.freeTensionBeltObject_OW(self, superFunc, objectId, objectsToJointTable, isDynamic, object)
    local _isSplitFree = (entityExists(objectId) and (getSplitType(objectId) ~= 0)) or false
    local entry = objectsToJointTable[objectId]
    local rmc = self.spec_rmcBeltAnchor

    if entry and entry.rmcMultiJoint then
        if self.isServer then
            local extras = entry.rmcMultiJoint
            local primary = entry.jointIndex
            for _, j in ipairs(extras.joints or {}) do
                if j ~= primary then pcall(removeJoint, j) end
            end
            for _, a in ipairs(extras.anchors or {}) do
                if a ~= entry.jointTransform and a and entityExists(a) then pcall(delete, a) end
            end
            if _isSplitFree then setInertiaScale(objectId,1,1,1) end
            if entry.rmcMassOrig and entityExists(objectId) then pcall(setMass, objectId, entry.rmcMassOrig) end
        end
        local res = superFunc(self, objectId, objectsToJointTable, isDynamic, object) -- base removes primary & resets flags
        if rmc and rmc.entries then rmc.entries[objectId] = nil; rmc.blockDM = next(rmc.entries) ~= nil end
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Free: %s", tostring(self.configFileName or self.typeName))) end
        return res
    end

    if entry and entry.rmcKinematic then
        if entry.object and entry.object.unmountKinematic then
            pcall(entry.object.unmountKinematic, entry.object, self)
        else
            if entityExists(objectId) then
                link(getRootNode(), objectId)
                setRigidBodyType(objectId, RigidBodyType.DYNAMIC)
            end
        end
        local res = superFunc(self, objectId, objectsToJointTable, isDynamic, object)
        if rmc and rmc.entries then rmc.entries[objectId] = nil end
        return res
    end

    if entry and entry.rmcInertiaApplied and entityExists(objectId) then pcall(setInertiaScale, objectId, 1,1,1) end

    return superFunc(self, objectId, objectsToJointTable, isDynamic, object)
end
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Loaded v1.1.0.0 (mass scaling on strap; default 0.85). Use 'gsRMCBeltAnchorMassScale <value>'.") end

-- Console handler
function RMCBeltAnchor:consoleSetObjHeightBias(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then v = tonumber(arg[1]) end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.OBJ_HEIGHT_BIAS = %.3f", RMCBeltAnchor.OBJ_HEIGHT_BIAS or 0))
        print("Usage: consoleSetObjHeightBias <value>")
        return
    end
    v = math.max(-0.1, math.min(0.1, v))
    RMCBeltAnchor.OBJ_HEIGHT_BIAS = v
    print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.OBJ_HEIGHT_BIAS set to %.3f", v))
end


-- Console handler
function RMCBeltAnchor:consoleSetObjAnchorSpread(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then v = tonumber(arg[1]) end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.OBJ_ANCHOR_SPREAD = %.3f", RMCBeltAnchor.OBJ_ANCHOR_SPREAD or 0))
        print("Usage: consoleSetObjAnchorSpread <value>")
        return
    end
    v = math.max(0.1, math.min(0.5, v))
    RMCBeltAnchor.OBJ_ANCHOR_SPREAD = v
    print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.OBJ_ANCHOR_SPREAD set to %.3f", v))
end


function RMCBeltAnchor:consoleSetObjClampDownOnly(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then v = tonumber(arg[1]) end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] OBJ_CLAMP_DOWN_ONLY = %d", RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY or 1))
        print("Usage: gsRMCObjClampDownOnly <0|1>")
        return
    end
    v = (v ~= 0) and 1 or 0
    RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY = v
    print(string.format("[RMC_BeltAnchor] OBJ_CLAMP_DOWN_ONLY set to %d", v))
end


-- Console handler
function RMCBeltAnchor:consoleSetObjMassScale(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then v = tonumber(arg[1]) end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.OBJ_MASS_SCALE = %.3f", RMCBeltAnchor.OBJ_MASS_SCALE or 0))
        print("Usage: consoleSetObjMassScale <value>")
        return
    end
    v = math.max(0.1, math.min(3.0, v))
    RMCBeltAnchor.OBJ_MASS_SCALE = v
    print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.OBJ_MASS_SCALE set to %.3f", v))
end


function RMCBeltAnchor:consoleSetObjInertiaScale(arg1, arg2, arg3)
    local sx, sy, sz = nil, nil, nil
    -- GIANTS forwards the whole line as a single string to first arg; handle both
    if type(arg1) == "string" and arg2 == nil and arg3 == nil then
        local a,b,c = string.match(arg1 or "", "([%-%d%.]+)%s+([%-%d%.]+)%s+([%-%d%.]+)")
        if a and b and c then sx, sy, sz = tonumber(a), tonumber(b), tonumber(c) end
    end
    if sx == nil then
        sx = tonumber(arg1); sy = tonumber(arg2); sz = tonumber(arg3)
    end
    if sx == nil or sy == nil or sz == nil then
        print(string.format("[RMC_BeltAnchor] OBJ_INERTIA_S* = %.2f %.2f %.2f", RMCBeltAnchor.OBJ_INERTIA_SX or 1, RMCBeltAnchor.OBJ_INERTIA_SY or 1, RMCBeltAnchor.OBJ_INERTIA_SZ or 1))
        print("Usage: gsRMCObjInertiaScale <sx> <sy> <sz>")
        return
    end
    sx = math.max(0.10, math.min(5.00, sx))
    sy = math.max(0.10, math.min(5.00, sy))
    sz = math.max(0.10, math.min(5.00, sz))
    RMCBeltAnchor.OBJ_INERTIA_SX, RMCBeltAnchor.OBJ_INERTIA_SY, RMCBeltAnchor.OBJ_INERTIA_SZ = sx, sy, sz
    print(string.format("[RMC_BeltAnchor] OBJ_INERTIA_S* set to %.2f %.2f %.2f", sx, sy, sz))
end


-- Console handler
function RMCBeltAnchor:consoleSetMaxJointCount(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then v = tonumber(arg[1]) end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.MAX_JOINT_COUNT = %.0f", RMCBeltAnchor.MAX_JOINT_COUNT or 0))
        print("Usage: consoleSetMaxJointCount <value>")
        return
    end
    v = math.max(1, math.min(3, v))
    RMCBeltAnchor.MAX_JOINT_COUNT = v
    print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.MAX_JOINT_COUNT set to %.0f", v))
end


-- Console handler
function RMCBeltAnchor:consoleSetMinMassForTri(arg)
    local v = nil
    if arg ~= nil then
        if type(arg) == "string" and arg ~= "" then v = tonumber(arg)
        elseif type(arg) == "table" and arg[1] ~= nil then v = tonumber(arg[1]) end
    end
    if v == nil then
        print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.MIN_MASS_FOR_TRI = %.1f", RMCBeltAnchor.MIN_MASS_FOR_TRI or 0))
        print("Usage: consoleSetMinMassForTri <value>")
        return
    end
    v = math.max(0, math.min(10000, v))
    RMCBeltAnchor.MIN_MASS_FOR_TRI = v
    print(string.format("[RMC_BeltAnchor] RMCBeltAnchor.MIN_MASS_FOR_TRI set to %.1f", v))
end


function RMCBeltAnchor:consoleApplyPreset(arg)
    local name = nil
    if type(arg) == "string" and arg ~= "" then name = arg
    elseif type(arg) == "table" and arg[1] ~= nil then name = tostring(arg[1]) end
    if not name then
        print("Usage: gsRMCPreset <TightLock|SmoothHaul|LightCargo>")
        return
    end
    name = string.lower(name)
    if name == "tightlock" then
        RMCBeltAnchor.OBJ_DECK_CLEARANCE = 0.035
        RMCBeltAnchor.OBJ_HEIGHT_BIAS    = -0.010
        RMCBeltAnchor.OBJ_ANCHOR_SPREAD  = 0.26
        RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY= 1
        RMCBeltAnchor.OBJ_MASS_SCALE     = 0.95
        RMCBeltAnchor.OBJ_INERTIA_SX, RMCBeltAnchor.OBJ_INERTIA_SY, RMCBeltAnchor.OBJ_INERTIA_SZ = 1.10, 1.10, 1.10
        RMCBeltAnchor.MAX_JOINT_COUNT    = 3
        RMCBeltAnchor.MIN_MASS_FOR_TRI   = 300
        print("[RMC_BeltAnchor] Preset TightLock applied")
    elseif name == "smoothhaul" then
        RMCBeltAnchor.OBJ_DECK_CLEARANCE = 0.045
        RMCBeltAnchor.OBJ_HEIGHT_BIAS    = 0.000
        RMCBeltAnchor.OBJ_ANCHOR_SPREAD  = 0.22
        RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY= 1
        RMCBeltAnchor.OBJ_MASS_SCALE     = 1.00
        RMCBeltAnchor.OBJ_INERTIA_SX, RMCBeltAnchor.OBJ_INERTIA_SY, RMCBeltAnchor.OBJ_INERTIA_SZ = 1.00, 1.00, 1.00
        RMCBeltAnchor.MAX_JOINT_COUNT    = 3
        RMCBeltAnchor.MIN_MASS_FOR_TRI   = 250
        print("[RMC_BeltAnchor] Preset SmoothHaul applied")
    elseif name == "lightcargo" then
        RMCBeltAnchor.OBJ_DECK_CLEARANCE = 0.040
        RMCBeltAnchor.OBJ_HEIGHT_BIAS    = 0.000
        RMCBeltAnchor.OBJ_ANCHOR_SPREAD  = 0.18
        RMCBeltAnchor.OBJ_CLAMP_DOWN_ONLY= 1
        RMCBeltAnchor.OBJ_MASS_SCALE     = 1.00
        RMCBeltAnchor.OBJ_INERTIA_SX, RMCBeltAnchor.OBJ_INERTIA_SY, RMCBeltAnchor.OBJ_INERTIA_SZ = 1.00, 1.00, 1.00
        RMCBeltAnchor.MAX_JOINT_COUNT    = 2
        RMCBeltAnchor.MIN_MASS_FOR_TRI   = 400
        print("[RMC_BeltAnchor] Preset LightCargo applied")
    else
        print("Unknown preset. Use: TightLock | SmoothHaul | LightCargo")
    end
end
